//-----------------------------------------------------------------------------
// VST Plug-Ins SDK
// User interface framework for VST plugins
//
// Version 2.0 Beta 4       Date : 21/03/00
//
// First version            : Wolfgang Kundrus         06.97
// Added Motif/Windows vers.: Yvan Grabit              01.98
// Added Mac version        : Charlie Steinberg        02.98
// Added BeOS version       : Georges-Edouard Berenger 05.99
//
// (c)1999-2000 Steinberg Soft+Hardware GmbH, All Rights Reserved
//-----------------------------------------------------------------------------

#ifndef __vstgui__
#define __vstgui__

// define global defines
#if WIN32
	#define WINDOWS 1
#elif SGI | SUN
	#define MOTIF 1
#elif __MWERKS__
	#define MAC 1
#endif

#ifndef __AEffEditor__
#include "AEffEditor.hpp"
#endif


#define VSTGUI_VERSION_MAJOR  2
#define VSTGUI_VERSION_MINOR  0

//----------------------------------------------------
#if WINDOWS
	#include <windows.h>

//----------------------------------------------------
#elif MOTIF
	#include <X11/Xlib.h>
	#include <X11/Intrinsic.h>
	#ifdef NOBOOL
		#ifndef bool
			typedef short bool;
		#endif
		#ifndef false
			static const bool false = 0; 
		#endif
		#ifndef true
			static const bool true = 1;
		#endif
	#endif

	// definition of struct for XPixmap resources
	struct CResTableEntry {
		int id;
  		char **xpm;
	};

	typedef CResTableEntry CResTable[];
	extern CResTable xpmResources;

//----------------------------------------------------
#elif MAC
	#include <Quickdraw.h>
	#include <Menus.h>
	#include <Windows.h>
	#include <TextUtils.h>
	#include <ToolUtils.h>
	#include <Resources.h>
	class CCashedPict;

//----------------------------------------------------
#elif BEOS
	#include <Font.h>
	class BView;
	class PlugView;
	class BBitmap;
	class BResources;

#endif

//----------------------------------------------------
//----------------------------------------------------
class CFrame;
class CDrawContext;
class COffscreenContext;
class CControl;
class CBitmap;


//-----------------------------------------------------------------------------
// AEffGUIEditor Declaration
//-----------------------------------------------------------------------------
class AEffGUIEditor : public AEffEditor
{
public :

	AEffGUIEditor (AudioEffect *pEffect);

	virtual ~AEffGUIEditor ();

	virtual void setParameter (long index, float value) { postUpdate (); } 
	virtual long getRect (ERect **ppRect);
	virtual long open (void *ptr);
	virtual void idle ();
	virtual void draw (ERect *pRect);
#if MAC
	virtual long mouse (long x, long y);
#endif

	// wait (in ms)
	void wait (unsigned long ms);

	// get the current time (in ms)
	unsigned long getTicks ();

	// feedback to appli.
	void doIdleStuff ();

	// get the effect attached to this editor
	AudioEffect *getEffect () { return effect; }

	// get version of this VSTGUI
	int getVstGuiVersion () { return VSTGUI_VERSION_MAJOR << 16 + VSTGUI_VERSION_MINOR; }

//---------------------------------------
protected:
	ERect   rect;
	CFrame *frame;

private:
	unsigned long lLastTicks;
	bool inIdleStuff;
};


//-----------------------------------------------------------------------------
// Structure CRect
//-----------------------------------------------------------------------------
struct CRect
{
	CRect (long left = 0, long top = 0, long right = 0, long bottom = 0)
	:	left (left), top (top), right (right), bottom (bottom) {}
	CRect (const CRect& r)
	:	left (r.left), top (r.top), right (r.right), bottom (r.bottom) {}
	CRect& operator () (long left, long top, long right, long bottom)
	{
		if (left < right)
			this->left = left, this->right = right;
		else
			this->left = right, this->right = left;
		if (top < bottom)
			this->top = top, this->bottom = bottom;
		else
			this->top = bottom, this->bottom = top;
		return *this;
	}

	bool operator != (const CRect& other) const
	{ return (left != other.left || right != other.right ||
				top != other.top || bottom != other.bottom); }

	bool operator == (const CRect& other) const
	{ return (left == other.left && right == other.right &&
				top == other.top && bottom == other.bottom); }
	
	inline long width ()  { return right - left; }
	inline long height () { return bottom - top; }

	CRect &offset (long x, long y)
	{ left += x; right += x; top += y; bottom += y; return *this; }

	CRect &inset (long deltaX, long deltaY)
	{ left += deltaX; right -= deltaX; top += deltaY; bottom -= deltaY;
    return *this; }
	
	long left;
	long top;
	long right;
	long bottom;
};

//-----------------------------------------------------------------------------
// Structure CPoint
//-----------------------------------------------------------------------------
struct CPoint
{
	CPoint (long h = 0, long v = 0) : h (h), v (v) {}
	CPoint& operator () (long h, long v) 
	{ this->h = h; this->v = v; return *this; }

	bool isInside (CRect& r)
	{ return h >= r.left && h <= r.right && v >= r.top && v <= r.bottom; } 

	bool operator != (const CPoint &other) const
	{ return (h != other.h || v != other.v); }

	bool operator == (const CPoint &other) const
	{ return (h == other.h && v == other.v); }

	CPoint &offset (long h, long v)
	{ this->h += h; this->v += v; return *this; }

	long h;
	long v;
};

//-----------------------------------------------------------------------------
// Structure CColor
//-----------------------------------------------------------------------------
struct CColor
{
	CColor& operator () (unsigned char red,
						unsigned char green,
						unsigned char blue,
						unsigned char unused)
	{
		this->red   = red;
		this->green = green;
		this->blue  = blue;
		this->unused = unused;
		return *this; 
	}

	CColor& operator = (CColor newColor)
	{
		red   = newColor.red;
		green = newColor.green;
		blue  = newColor.blue;
		unused = newColor.unused;
		return *this; 
	}

	bool operator != (const CColor &other) const 
	{ return (red != other.red || green != other.green || blue  != other.blue); }

	bool operator == (const CColor &other) const
	{ return (red == other.red && green == other.green && blue  == other.blue); }
	
	unsigned char red;
	unsigned char green;
	unsigned char blue;
	unsigned char unused;
};

// define some basic colors
static CColor kTransparentCColor = {255, 255, 255, 0};
static CColor kBlackCColor  = {0,     0,   0, 0};
static CColor kWhiteCColor  = {255, 255, 255, 0};
static CColor kGreyCColor   = {127, 127, 127, 0};
static CColor kRedCColor    = {255,   0,   0, 0};
static CColor kGreenCColor  = {0  , 255,   0, 0};
static CColor kBlueCColor   = {0  ,   0, 255, 0};
static CColor kYellowCColor = {255, 255,   0, 0};
static CColor kCyanCColor   = {255,   0, 255, 0};
static CColor kMagentaCColor= {0  , 255, 255, 0};


//-----------------------------------------------------------------------------
// Definitions of special characters in a platform independent way

#if WINDOWS || MOTIF
#define kDegreeSymbol      "\xB0"
#define kInfiniteSymbol    "oo"
#define kCopyrightSymbol   "\xA9"
#define kTrademarkSymbol   "\x99"
#define kRegisteredSymbol  "\xAE"
#define kMicroSymbol       "\x85"
#define kPerthousandSymbol "\x89"

#elif BEOS
#define kDegreeSymbol      "\xC2\xB0"
#define kInfiniteSymbol    "\xE2\x88\x9E"
#define kCopyrightSymbol   "\xC2\xA9"
#define kTrademarkSymbol   "\xE2\x84\xA2"
#define kRegisteredSymbol	 "\xC2\xAE"
#define kMicroSymbol       "\xC2\xB5"
#define kPerthousandSymbol "\xE2\x80\xB0"

#elif MAC
#define kDegreeSymbol      "\A1"
#define kInfiniteSymbol    "\B0"
#define kCopyrightSymbol   "\A9"
#define kTrademarkSymbol   "\AA"
#define kRegisteredSymbol  "\A8"
#define kMicroSymbol       "\B5"
#define kPerthousandSymbol "\E4"
#endif

//-----------------------------------------------------------------------------
//-----------
// Font Type
//-----------
enum CFont
{
	kSystemFont = 0,
	kNormalFontVeryBig,
	kNormalFontBig,
	kNormalFont,
	kNormalFontSmall,
	kNormalFontSmaller,
	kNormalFontVerySmall,
	kSymbolFont,

	kNumStandardFonts
};

//-----------
// Line Style
//-----------
enum CLineStyle
{
	kLineSolid = 0,
	kLineOnOffDash
};

//-----------
// Draw Mode
//-----------
enum CDrawMode
{
	kCopyMode = 0,
	kOrMode,
	kXorMode
};

//----------------------------
// Text Alignment (Horizontal)
//----------------------------
enum CHoriTxtAlign
{
	kLeftText = 0,
	kCenterText,
	kRightText
};

//----------------------------
// Buttons Type (+modifiers)
//----------------------------
enum CButton
{
	kLButton =  1,
	kMButton =  2,
	kRButton =  4,
	kShift   =  8,
	kControl = 16,
	kAlt     = 32,
	kApple   = 64
};

//-----------------------------------------------------------------------------
// CDrawContext Declaration
//-----------------------------------------------------------------------------
class CDrawContext
{
public:
	CDrawContext (CFrame *pFrame, void *pSystemContext, void *pWindow = 0);
	virtual ~CDrawContext ();	

	void moveTo (CPoint &point);
	void lineTo (CPoint &point);

	void polyLine (CPoint *pPoint, int numberOfPoints);
	void fillPolygon (CPoint *pPoint, int numberOfPoints);

	void drawRect (CRect &rect);
	void fillRect (CRect &rect);

	void drawArc (CRect &rect, CPoint &point1, CPoint &point2);
	void fillArc (CRect &rect, CPoint &point1, CPoint &point2);

	void drawEllipse (CRect &rect);
	void fillEllipse (CRect &rect);
	
	void drawPoint (CPoint &point, CColor color);
	
	void       setLineStyle (CLineStyle style);
	CLineStyle getLineStyle () { return lineStyle; }

	void   setLineWidth (int width);
	int    getLineWidth () { return frameWidth; }

	void      setDrawMode (CDrawMode mode);
	CDrawMode getDrawMode () { return drawMode; }

	void   setClipRect (const CRect &clip);
	CRect &getClipRect (CRect &clip) { clip = clipRect; return clip; }
	void   resetClipRect ();

	void   setFillColor  (CColor color);
	CColor getFillColor () { return fillColor; }

	void   setFrameColor (CColor color);
	CColor getFrameColor () { return frameColor; }

	void   setFontColor (CColor color);
	CColor getFontColor () { return fontColor; }
	void   setFont (CFont fontID, const int size = 0);

	long getStringWidth (const char* pStr);

	void drawString (const char *pString, CRect &rect, const short opaque = false,
					 const CHoriTxtAlign hAlign = kCenterText);

	int  getMouseButtons ();
	void getMouseLocation (CPoint &point);

#if MOTIF
	int getIndexColor (CColor color);
	Colormap getColormap ();
	Visual   *getVisual ();
	unsigned int getDepth ();

	static int nbNewColor;
#endif

	void *getWindow () { return pWindow; }
	void setWindow (void *ptr)  { pWindow = ptr; }
	void getLoc (CPoint &where) { where = penLoc; }

	//-------------------------------------------
protected:

	friend class CBitmap;
	friend class COffscreenContext;

	void   *getSystemContext () { return pSystemContext; }

	void   *pSystemContext;
	void   *pWindow;
	CFrame *pFrame;

	int    fontSize;
	CColor fontColor;
	CPoint penLoc;

	int    frameWidth;
	CColor frameColor;
	CColor fillColor;
	CLineStyle lineStyle;
	CDrawMode  drawMode;
	CRect  clipRect;

#if WINDOWS
	void *pBrush;
	void *pPen;
	void *pFont;
	void *pOldBrush;
	void *pOldPen;
	void *pOldFont;
	int  iPenStyle;

#elif MAC
	FontInfo fontInfoStruct;
	Pattern  fillPattern;

#elif MOTIF
	Display *pDisplay;

	XFontStruct *pFontInfoStruct;
	CFont fontInfoId;

#elif BEOS
	BView*	pPlugView;
	BFont	font;
	void lineFromTo (CPoint& cstart, CPoint& cend);

#endif
};

//-----------------------------------------------------------------------------
// COffscreenContext Declaration
//-----------------------------------------------------------------------------
class COffscreenContext : public CDrawContext
{
public:
	COffscreenContext (CDrawContext *pContext, CBitmap *pBitmap);
	COffscreenContext (CFrame *pFrame, long width, long height, const CColor backgroundColor = kBlackCColor);

	virtual ~COffscreenContext ();

	// Old interface (for compatibility)
	void transfert (CDrawContext *pContext, CRect destRect, CPoint srcOffset = CPoint (0, 0));

	void copyFrom (CDrawContext *pContext, CRect destRect, CPoint srcOffset = CPoint (0, 0));

	inline int getWidth ()  { return width; }
	inline int getHeight () { return height; }

	//-------------------------------------------
protected:
	bool    destroyPixmap;
	CBitmap *pBitmap;

	long    height;
	long    width;

	CColor  backgroundColor;

#if MOTIF
	Display *pXdisplay;

#elif BEOS
	BBitmap *offscreenBitmap;
#endif
};


//-----------------------------------------------------------------------------
// CBitmap Declaration
//-----------------------------------------------------------------------------
class CBitmap
{
public:
	CBitmap (int resourceID);
	CBitmap (CFrame &frame, int width, int height);
	~CBitmap ();

	void draw (CDrawContext*, CRect &rect, const CPoint &offset = CPoint (0, 0));
	void drawTransparent (CDrawContext *pContext, CRect &rect, const CPoint &offset = CPoint (0, 0));

	inline int getWidth ()  { return width; }
	inline int getHeight () { return height; }

	void forget ();
	void remember ();
	int  getNbReference () { return nbReference; }

	void *getHandle () { return pHandle; }
	
#if BEOS
	static void closeResource ();
#endif

	//-------------------------------------------
protected:
	int  resourceID;
	void *pHandle;
	void *pMask;
	int  nbReference;
	int  width;
	int  height;

#if MOTIF
	void *createPixmapFromXpm (CDrawContext *pContext);

	char    **ppDataXpm;
	Display *pXdisplay;

#elif MAC
	CCashedPict	*pPict;

#elif BEOS
	static BResources *resourceFile;
	BBitmap    *bbitmap;
	bool		transparencySet;
#endif
};

//-----------------------------------------------------------------------------
// CView Declaration
//-----------------------------------------------------------------------------
class CView
{
public:
	CView (CRect &size);
	virtual ~CView ();

	void redraw ();
	virtual void draw (CDrawContext *pContext);
	virtual void mouse (CDrawContext *pContext, CPoint &where);
	virtual void update (CDrawContext *pContext);

	virtual void looseFocus ();
	virtual void takeFocus ();

	virtual bool isDirty () { return bDirty; }
	virtual void setDirty (const bool val = true) { bDirty = val; }

	virtual void setMouseEnabled (const bool bEnable = true) { bMouseEnabled = bEnable; }
	virtual bool getMouseEnabled () { return bMouseEnabled; }

	virtual void setMouseableArea (const CRect &rect)  { mouseableArea = rect; }
	virtual CRect &getMouseableArea (CRect &rect) { rect = mouseableArea; return rect;}

	virtual void setTransparency (bool val) { bTransparencyEnabled = val; }
	virtual bool getTransparency () { return bTransparencyEnabled; }

	int getHeight () { return size.height (); }
	int getWidth ()  { return size.width (); }
	CRect &getViewSize (CRect &rect) { rect = size; return rect; }

	CFrame *getParent () { return pParent; }

	//-------------------------------------------
protected:
	friend class CControl;
	friend class CFrame;
	friend class CViewContainer;

	CRect  size;
	CRect  mouseableArea;

	CFrame *pParent;

	bool  bDirty;
	bool  bMouseEnabled;
	bool  bTransparencyEnabled;
};

//-----------------------------------------------------------------------------
// CFrame Declaration
//-----------------------------------------------------------------------------
class CFrame : public CView
{
public:
	CFrame (CRect &size, void *pSystemWindow, AEffGUIEditor *pEditor);
	CFrame (CRect &size, char *pTitle, AEffGUIEditor *pEditor, const int style = 0);
	
	~CFrame ();

	bool open (CPoint *pPoint = 0);
	bool close ();
	bool isOpen () { return openFlag; }

	void draw (CDrawContext *pContext);
	void draw (CView *pView = 0);
	void mouse (CDrawContext *pContext, CPoint &where);

	void update (CDrawContext *pContext);
	void idle ();
	void doIdleStuff () { if (pEditor) pEditor->doIdleStuff (); }

	bool getPosition (int &x, int &y);
	bool setSize (int width, int height);
	bool getSize (CRect *pSize);

	void     setBackground (CBitmap *pBackground);
	CBitmap *getBackground () { return pBackground; }

	virtual bool addView (CView *pView);
	virtual bool removeView (CView *pView);
	virtual bool isChild (CView *pView);

	int   setModalView (CView *pView);
 
#if WINDOWS
	HWND getOuterWindow ();
	void *getSystemWindow () { return pHwnd;}
#elif BEOS
	void *getSystemWindow () { return pPlugView;}
#else
	void *getSystemWindow () { return pSystemWindow;}
#endif

	AEffGUIEditor *getEditor () { return pEditor; }

	void   setEditView (CView *pView) { pEditView = pView; }
	CView *getEditView () { return pEditView; }

#if MOTIF
	Colormap getColormap ()   { return colormap; }
	Visual  *getVisual ()     { return pVisual; }
	unsigned int getDepth ()  { return depth; }
	Display *getDisplay ()    { return pDisplay; }
	Window   getWindow ()     { return window; }
	void     freeGc ();

	Region   region;

	GC       gc;
	GC       getGC ()         { return gc; }
#else
	void    *getGC ()         { return 0; }
#endif

	//-------------------------------------------
protected:
	bool   initFrame (void *pSystemWin);

	AEffGUIEditor *pEditor;
	
	void    *pSystemWindow;
	CBitmap *pBackground;
	int      viewCount;
	int      maxViews;
	CView   **ppViews;
	CView   *pModalView;
	CView   *pEditView;

	bool    firstDraw;
	bool    openFlag;

#if WINDOWS
	void    *pHwnd;
	friend LONG WINAPI WindowProc (HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam);

#elif MOTIF
	Colormap  colormap;
	Display  *pDisplay;
	Visual   *pVisual;
	Window    window;
	unsigned int depth;

	friend void _destroyCallback (Widget, XtPointer, XtPointer);

#elif BEOS
	PlugView *pPlugView;
#endif

	//-------------------------------------------
private:
	bool      addedWindow;
	void     *pVstWindow;
};


//-----------------------------------------------------------------------------
// CViewContainer Declaration
//-----------------------------------------------------------------------------
class CCView;
class CViewContainer : public CView
{
public:
	CViewContainer (CRect &size, CFrame *pParent, CBitmap *pBackground = 0, bool offscreen = false);
	~CViewContainer ();

	virtual void addView (CView *pView);
	virtual void addView (CView *pView, CRect &mouseableArea, bool mouseEnabled = true);
	virtual void removeView (CView *pView);
	virtual void removeAll ();
	virtual bool isChild (CView *pView);

	virtual void draw (CDrawContext *pContext);
	virtual void mouse (CDrawContext *pContext, CPoint &where);
	virtual void update (CDrawContext *pContext);

	virtual void looseFocus ();
	virtual void takeFocus ();

	//-------------------------------------------
protected:
	CCView  *pFirstView;
	CCView  *pLastView;
	CBitmap *pBackground;

	bool    offscreen;
};


// include the control objects
#ifndef __vstcontrols__
#include "vstcontrols.h"
#endif


//-End--------------------------------------
#endif

